<?php

namespace App\Http\Controllers\Api\vendor;

use Carbon\Carbon;
use App\Models\RFQ;
use App\Models\Order;
use App\Models\RfqBid;
use App\Models\Product;
use App\helpers\Attachment;
use App\Services\FcmService;
use Illuminate\Http\Request;
use App\Traits\HasCrudActions;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Http\Requests\Api\vendor\BidRequest;
use App\Http\Requests\Api\vendor\RfqRequest;
use App\Http\Resources\Api\vendor\RfqResource;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Resources\Api\vendor\OrderResource;

class RfqController extends Controller
{
    use HasCrudActions;


    protected $model = RFQ::class;

    protected $resource = RfqResource::class;

    protected $request = RfqRequest::class;
    public function index()
    {
        // Get the cities of the authenticated vendor's branches
        $branchCities = auth()->user()->branches->pluck('city_id');

        // Query the RFQs where the branch city matches the vendor's branch cities
        $modelResource = $this->search()
            ->whereHas('branch', function ($query) use ($branchCities) {
                $query->whereIn('city_id', $branchCities); // Filter by city
            })
            ->where(['vendor_id' => null, 'is_approved' => 0, 'order_type' => 'rfq_order', 'status' => 'pending'])
            ->whereDoesntHave('bids', function ($query) {
                $query->where('vendor_id', auth()->user()->id); // Exclude RFQs with vendor's bids
            })
            ->with([
                'product:id,name',
                'specs:id,key,value',
                'branch:id,name,location,city_id'
            ])
            ->orderBy('id', 'DESC') // Sort by most recent RFQ
            ->get();

        return $this->apiResponse('200', __('success'), '', RfqResource::collection($modelResource));
    }



    public function update(string $id, FcmService $fcmService)
    {
        try {
            $request = app($this->request);
    
            $record = $this->model::find($id);
    
            if (!$record) {
                return $this->apiResponse(Response::HTTP_OK, 'fail', __('resource not found'));
            }
    
            DB::beginTransaction();
    
            // Check if the status has changed
            $oldStatus = $record->status;
            $newStatus = $request->get('status');
            
            // Update the record except for the image
            $record->update($request->except('image'));
    
            DB::commit();
    
            // Send notification if status has changed
            if ($newStatus && $oldStatus !== $newStatus) {
                $user = $record->user; // Assuming the model has a `user` relation
                if ($user && $user->fcmTokens()->exists()) {
                    $fcmService->send(
                        $user->fcmTokens[0]->device_token,
                        __('Order Status Updated'),
                        __('Your order status has been changed to :status', ['status' => $newStatus]),
                        [
                            'record_id' => (string) $record->id,
                            'status' => $newStatus,
                            'action' => 'order_status_update'
                        ],
                        $user
                    );
                }
            }
    
            $recordResource = new $this->resource($record);
    
            return $this->apiResponse(Response::HTTP_OK, __('success'), __('Updated Successfully!'), $recordResource);
        } catch (\Exception $e) {
            DB::rollBack();
    
            $this->log(0, $this->model, $e->getMessage(), $e);
    
            return $this->errorMessage($e);
        }
    }
    

    public function myRfqs(Request $request)
    {
        $modelResource = $this->search();
    
        // Check if the 'status' filter is provided
        if ($request->has('status') && $request->status === 'completed') {
            // Filter by 'completed' status
            $modelResource = $modelResource->where('status', 'completed');
        } else {
            // If no 'status' filter, exclude 'completed' RFQs
            if ($request->has('date')) {
                $date = Carbon::parse($request->date)->format('Y-m-d'); // Ensure the date format
                $modelResource = $modelResource->whereDate('created_at', $date)->where('status', '!=', 'completed'); // Filter by specific date and exclude 'completed'
            } else {
                // Default to today and exclude 'completed'
                $today = Carbon::today()->format('Y-m-d');
                $modelResource = $modelResource->whereDate('created_at', $today)->where('status', '!=', 'completed');
            }
        }
    
        // Fetch and group the RFQs by the day they were created
        $rfqs = $modelResource->where(['vendor_id' => Auth::id(), 'order_type' => 'rfq_order'])
            ->with([
                'product:id,name,price',
                'specs:id,key,value',
                'branch:id,name,location'
            ])
            ->orderBy('id', 'DESC')
            ->get()
            ->makeHidden(['image'])
            ->groupBy(function ($rfq) {
                return $rfq->created_at->format('Y-m-d'); // Group by day
            });
    
        // Transform the grouped RFQs using RfqResource
        $transformedRfqs = $rfqs->map(function ($groupedRfqs, $date) {
            return [
                'date' => $date,
                'rfqs' => RfqResource::collection($groupedRfqs),
            ];
        });
    
        return $this->ApiResponse(
            '200',
            __('success'),
            '',
            $transformedRfqs->values() // Return the transformed and grouped data
        );
    }
    



    public function show(string $id)
    {
        $record = $this->model::find($id)->load([
            'product:id,name',
            'specs:id,key,value',
            'branch:id,name,location'
        ]);

        if (!$record) {
            return $this->apiResponse(Response::HTTP_OK, 'fail', __('resource not found'));
        }

        $recordResource = new $this->resource($record);

        return $this->apiResponse(Response::HTTP_OK, __('success'), '', $recordResource);
    }

    public function addBid(BidRequest $request, FcmService $fcmService)
    {
        try {
            // Begin transaction
            DB::beginTransaction();
    
            // Create the bid
            $model = RfqBid::create($request->all());
    
            // Find the associated RFQ and update the user_id
            $rfq = RFQ::find($request->rfq_id);
            if (!$rfq) {
                return $this->apiResponse(Response::HTTP_NOT_FOUND, __('error'), __('RFQ not found.'));
            }
    
            $model->user_id = $rfq->user_id;
            $rfq->save();
            $model->save();
    
            // Commit transaction
            DB::commit();
    
            // Send notification to the RFQ owner
            if ($rfq->user && $rfq->user->fcmTokens()->exists()) {
                $fcmService->send(
                    $rfq->user->fcmTokens[0]->device_token ?? null,
                    __('New Bid Received'),
                    __('A new bid has been placed on your RFQ.'),
                    [
                        'rfq_id' => (string) $rfq->id,
                        'bid_id' => (string) $model->id,
                        'action' => 'new_bid'
                    ],
                    $rfq->user
                );
            }
    
            return $this->apiResponse(Response::HTTP_OK, __('success'), __('Bid added successfully!'), $model);
        } catch (\Exception $e) {
            // Rollback transaction in case of an error
            DB::rollBack();
    
            // Log the exception
            $this->log(0, $this->model, $e->getMessage(), $e);
    
            return $this->errorMessage($e);
        }
    }
    

    public function statistics()
    {
        // Get all approved RFQs for the authenticated vendor
        $rfqs = RFQ::where(['vendor_id' => Auth::id(), 'is_approved' => 1])->get();

        // Calculate the total number of products sold
        $totalSoldProducts = $rfqs->sum('qty');

        // Get the top sold product by grouping the RFQs by product_id and summing the quantities
        $topSoldProduct = RFQ::where(['vendor_id' => Auth::id(), 'is_approved' => 1])
            ->select('product_id', DB::raw('SUM(qty) as total_quantity'))
            ->groupBy('product_id')
            ->orderBy('total_quantity', 'desc')
            ->first();

        // Get the product details for the top sold product
        $topProduct = null;
        if ($topSoldProduct) {
            $topProduct = Product::find($topSoldProduct->product_id);
        }

        // Get the total bids won and lost
        $totalBidsWon = RfqBid::where(['vendor_id' => Auth::id(), 'is_approved' => 1])->count();
        $totalBidsLost = RfqBid::where(['is_approved' => 0])->count();

        // Calculate the total revenue
        $totalRevenue = 0;
        foreach ($rfqs as $rfq) {
            // Assuming RFQ has a product relationship and product has a price field
            $product = Product::find($rfq->product_id);  // Fetch the product
            if ($product) {
                $totalRevenue += $rfq->qty * $product->price; // Calculate revenue for each RFQ
            }
        }

        return $this->apiResponse(
            '200',
            __('success'),
            '',
            [
                'total_sold_products' => $totalSoldProducts,
                'top_sold_product' => [
                    'id' => $topProduct? $topProduct->id : null,
                    'name' => $topProduct? $topProduct->name : null,
                ],
                'bids_won' => $totalBidsWon,
                'bids_lost' => $totalBidsLost,
                'total_revenue' => $totalRevenue,  // Add total revenue to the response
            ]
        );
    }

}
