<?php

namespace App\Http\Controllers\Api\Dentist;

use App\Models\RFQ;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use App\Traits\HasCrudActions;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Http\Resources\Api\Dentist\ProductResource;
use App\Http\Resources\Api\vendor\CategoryResource;
use App\Http\Resources\Api\Admin\CategoryResource as AdminCategoryResource;

class HomeController extends Controller
{
    use HasCrudActions;


    public function home()
    {
        $categories = Category::whereNull('parent_id')->get();

        // Get the top sold products with their rfq_count
        $topProducts = Product::with('mainCategory')
        ->select('products.id', 'products.name', 'products.price', 'products.description', 'products.main_category_id')
        ->leftJoin('rfq_items', 'products.id', '=', 'rfq_items.product_id')
        ->leftJoin('r_f_q_s', 'products.id', '=', 'r_f_q_s.product_id')
        ->selectRaw('COUNT(rfq_items.product_id) + COUNT(r_f_q_s.product_id) as rfq_count')
        ->groupBy('products.id', 'products.name', 'products.price', 'products.description', 'products.main_category_id')
        ->having('rfq_count', '>', 0) // Only get products with a count greater than 0
        ->orderByDesc('rfq_count')
        ->limit(5)
        ->get();

        return $this->apiResponse('200', __('success'), '', [
            'categories' => AdminCategoryResource::collection($categories),
            'topProducts' => ProductResource::collection($topProducts), // Wrap in resource collection
        ]);
    }


    public function categories(Request $request){
        $categories = Category::whereNull('parent_id')->get();
        if($request->has('category_id')){
            $categories = Category::where('id', $request->category_id)->with(['children' , 'children.children', 'children.children', 'products'])->first();
            return $this->apiResponse('200', __('success'), '', CategoryResource::make($categories));

        }
        return $this->apiResponse('200', __('success'), '', CategoryResource::collection($categories));
    }

    public function getFav()
    {
        $user = Auth::user();
        if (!$user) {
            return $this->apiResponse('401', __('error'), __('messages.unauthorized'));
        }


        return $this->apiResponse('200', __('success'), '', ProductResource::collection($user->favoriteProducts));
    }




    public function toggleFavorite($id)
    {
        // Find the product
        $product = Product::findOrFail($id);
        $user = Auth::user();

        // Toggle favorite status: attach if not present, detach if already present
        $user->favoriteProducts()->toggle($id);

        // Check if the product is now in favorites
        $isFavorite = $user->favoriteProducts()->where('product_id', $id)->exists();

        // Return response based on the action performed
        if ($isFavorite) {
            return response()->json(['message' => 'Product added to favorites']);
        } else {
            return response()->json(['message' => 'Product removed from favorites']);
        }
    }

    public function showProduct($id) {
        $product = Product::with(['mainCategory', 'specs' , 'subCategory'])
            ->where('id', $id)
            ->first();

        if (!$product) {
            return $this->apiResponse('404', __('error'), __('messages.product_not_found'));
        }

        return $this->apiResponse('200', __('success'), '', new ProductResource($product));
    }


    public function getCategoryProducts(Request $request)
    {
        $products = Product::where('sub_category_id' , $request->category_id)->orWhere('main_category_id' , $request->category_id)->get();
        return $this->apiResponse('200', __('success'), '',  ProductResource::collection($products));

    }

    public function statistics()
    {
        $rfqs = RFQ::where(['user_id' => auth()->user()->id, 'status' => 'completed'])->get();
    
        return $this->apiResponse(
             '200',
            __('success'),
            '',
            [
                'total_spent' => $rfqs->sum('price'),
                'purchased_products' => $rfqs->count(),
            ]
        );
    }
    
    

}
